<?php
/**
 * Custom template tags for this theme.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Spotlight
 */

if ( ! function_exists( 'spotlight_paging_nav' ) ) :
/**
 * Display navigation to next/previous set of posts when applicable.
 *
 * @return void
 */
function spotlight_paging_nav() {
	// Don't print empty markup if there's only one page.
	if ( $GLOBALS['wp_query']->max_num_pages < 2 ) {
		return;
	}
	?>
	<nav class="navigation paging-navigation clearfix" role="navigation">
		<div class="paging-links">

			<?php if ( get_next_posts_link() ) : ?>
			<div class="paging-previous"><?php next_posts_link( __( '<span class="paging-nav">&larr;</span> Older posts', 'spotlight' ) ); ?></div>
			<?php endif; ?>

			<?php if ( get_previous_posts_link() ) : ?>
			<div class="paging-next"><?php previous_posts_link( __( 'Newer posts <span class="paging-nav">&rarr;</span>', 'spotlight' ) ); ?></div>
			<?php endif; ?>

		</div><!-- .nav-links -->
	</nav><!-- .navigation -->
	<?php
}
endif;

if ( ! function_exists( 'spotlight_post_nav' ) ) :
/**
 * Display navigation to next/previous post when applicable.
 *
 * @return void
 */
function spotlight_post_nav() {
	// Don't print empty markup if there's nowhere to navigate.
	$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
	$next     = get_adjacent_post( false, '', false );

	if ( ! $next && ! $previous ) {
		return;
	}
	?>
	<nav class="navigation post-navigation" role="navigation">
		<div class="nav-links">

			<?php previous_post_link( '%link', _x( '<span class="meta-nav">&larr;</span>', 'Previous post link', 'spotlight' ) ); ?>
			<?php next_post_link('%link', _x( '<span class="meta-nav">&rarr;</span>', 'Next post link',     'spotlight' ) ); ?>

		</div><!-- .nav-links -->
	</nav><!-- .navigation -->
	<?php
}
endif;

if ( ! function_exists( 'spotlight_comment' ) ) :
/**
 * Template for comments and pingbacks.
 *
 * Used as a callback by wp_list_comments() for displaying the comments.
 */
function spotlight_comment( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;

	if ( 'pingback' == $comment->comment_type || 'trackback' == $comment->comment_type ) : ?>

	<li id="comment-<?php comment_ID(); ?>" <?php comment_class(); ?>>
		<div class="comment-body">
			<?php _e( 'Pingback:', 'spotlight' ); ?> <?php comment_author_link(); ?> <?php edit_comment_link( __( 'Edit', 'spotlight' ), '<span class="edit-link">', '</span>' ); ?>
		</div>

	<?php else : ?>

	<li id="comment-<?php comment_ID(); ?>" <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ); ?>>
		<article id="div-comment-<?php comment_ID(); ?>" class="comment-body">
			<footer class="comment-meta">
				<div class="comment-author vcard">
					<?php if ( 0 != $args['avatar_size'] ) { echo get_avatar( $comment, $args['avatar_size'] ); } ?>
					<?php printf( __( '%s <span class="says">says:</span>', 'spotlight' ), sprintf( '<cite class="fn">%s</cite>', get_comment_author_link() ) ); ?>
				</div><!-- .comment-author -->

				<div class="comment-metadata">
					<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ); ?>">
						<time datetime="<?php comment_time( 'c' ); ?>">
							<?php printf( _x( '%1$s at %2$s', '1: date, 2: time', 'spotlight' ), get_comment_date(), get_comment_time() ); ?>
						</time>
					</a>
					<?php edit_comment_link( __( 'Edit', 'spotlight' ), '<span class="edit-link">', '</span>' ); ?>
				</div><!-- .comment-metadata -->

				<?php if ( '0' == $comment->comment_approved ) : ?>
				<p class="comment-awaiting-moderation"><?php _e( 'Your comment is awaiting moderation.', 'spotlight' ); ?></p>
				<?php endif; ?>
			</footer><!-- .comment-meta -->

			<div class="comment-content">
				<?php comment_text(); ?>
			</div><!-- .comment-content -->

			<?php
				comment_reply_link( array_merge( $args, array(
					'add_below' => 'div-comment',
					'depth'     => $depth,
					'max_depth' => $args['max_depth'],
					'before'    => '<div class="reply">',
					'after'     => '</div>',
				) ) );
			?>
		</article><!-- .comment-body -->

	<?php
	endif;
}
endif; // ends check for spotlight_comment()

if ( ! function_exists( 'spotlight_posted_on' ) ) :
/**
 * Prints HTML with meta information for the current post-date/time and author.
 */
function spotlight_posted_on() {
	$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time>';
	if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
		$time_string .= '<time class="updated" datetime="%3$s">%4$s</time>';
	}

	$time_string = sprintf( $time_string,
		esc_attr( get_the_date( 'c' ) ),
		esc_html( get_the_date() ),
		esc_attr( get_the_modified_date( 'c' ) ),
		esc_html( get_the_modified_date() )
	);

	printf( __( '<span class="posted-on">Posted on %1$s</span><span class="byline"> by %2$s</span>', 'spotlight' ),
		sprintf( '<a href="%1$s" rel="bookmark">%2$s</a>',
			esc_url( get_permalink() ),
			$time_string
		),
		sprintf( '<span class="author vcard"><a class="url fn n" href="%1$s">%2$s</a></span>',
			esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
			esc_html( get_the_author() )
		)
	);
}
endif;

/**
 * Post tags without link
 */
function the_spotlight_tags() { 
	$posttags = get_the_tags();
	if ($posttags) {
	  foreach($posttags as $tag) {
	    echo $tag->name . ', '; 
	  }
	}
}

function get_spotlight_tags() { 
	$posttags = get_the_tags();
	if ($posttags) {
	  foreach($posttags as $tag) {
	    return $tag->name . ', '; 
	  }
	}
}

/**
 * Returns true if a blog has more than 1 category.
 */
function spotlight_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'all_the_cool_cats' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'hide_empty' => 1,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'all_the_cool_cats', $all_the_cool_cats );
	}

	if ( '1' != $all_the_cool_cats ) {
		// This blog has more than 1 category so spotlight_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so spotlight_categorized_blog should return false.
		return false;
	}
}

/**
 * Flush out the transients used in spotlight_categorized_blog.
 */
function spotlight_category_transient_flusher() {
	// Like, beat it. Dig?
	delete_transient( 'all_the_cool_cats' );
}
add_action( 'edit_category', 'spotlight_category_transient_flusher' );
add_action( 'save_post',     'spotlight_category_transient_flusher' );

/**
 * Default homepage layout
 */

function default_homepage_layout( $arg1, $arg2 ) {
	$layout_options_array = array();
	
	switch ( $arg1 ) {
		// Full-width
		case 'full_width_normal_unequal_grid':
			$layout_options_array['container'] = '';
			$layout_options_array['cell-width'] = 380;
			$layout_options_array['thumbnail-size'] = 'full-width-normal-unequal-grid';
		break;

		case 'full_width_small_unequal_grid':
			$layout_options_array['container'] = '';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'full-width-small-unequal-grid';
		break;

		case 'full_width_mini_unequal_grid':
			$layout_options_array['container'] = '';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'full-width-mini-unequal-grid';
		break;

		case 'full_width_small_equal_grid':
			$layout_options_array['container'] = '';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'full-width-small-equal-grid';
		break;

		case 'full_width_mini_equal_grid':
			$layout_options_array['container'] = '';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'full-width-mini-equal-grid';
		break;

		// Container - 1170px
		case 'container_normal_unequal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 380;
			$layout_options_array['thumbnail-size'] = 'container-normal-unequal-grid';
		break;

		case 'container_small_unequal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'container-small-unequal-grid';
		break;

		case 'container_mini_unequal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'container-mini-unequal-grid';
		break;

		case 'container_normal_equal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 380;
			$layout_options_array['thumbnail-size'] = 'container-small-equal-grid';
		break;

		case 'container_small_equal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'container-small-equal-grid';
		break;

		case 'container_mini_equal_grid':
			$layout_options_array['container'] = 'container';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'container-mini-equal-grid';
		break;

		// Narrow - 970px
		case 'narrow_normal_unequal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 380;
			$layout_options_array['thumbnail-size'] = 'narrow-normal-unequal-grid';
		break;

		case 'narrow_small_unequal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'narrow-small-unequal-grid';
		break;

		case 'narrow_mini_unequal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'narrow-mini-unequal-grid';
		break;

		case 'narrow_normal_equal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 380;
			$layout_options_array['thumbnail-size'] = 'narrow-small-equal-grid';
		break;

		case 'narrow_small_equal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 280;
			$layout_options_array['thumbnail-size'] = 'narrow-mini-equal-grid';
		break;

		case 'narrow_mini_equal_grid':
			$layout_options_array['container'] = 'container narrow';
			$layout_options_array['cell-width'] = 210;
			$layout_options_array['thumbnail-size'] = 'narrow-mini-equal-grid';
		break;

	}

	switch ( $arg2 ) {
		case 'both':
			$layout_options_array['gutter-y'] = 100;
			$layout_options_array['container-margin-bottom'] = 100;
		break;
		case 'title':
			$layout_options_array['gutter-y'] = 70;
			$layout_options_array['container-margin-bottom'] = 70;
		break;
		case 'tags':
			$layout_options_array['gutter-y'] = 70;
			$layout_options_array['container-margin-bottom'] = 70;
		break;
		case 'none':
			$layout_options_array['gutter-y'] = 0;
			$layout_options_array['container-margin-bottom'] = 0;
		break;
	}

	return($layout_options_array);
}
